/*
 * Decompiled with CFR 0.152.
 */
package org.exist.storage.lock;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.text.DateFormat;
import java.util.Arrays;
import java.util.Date;
import java.util.Properties;
import org.apache.log4j.Logger;
import org.exist.storage.BrokerPool;
import org.exist.storage.lock.FileLockHeartBeat;
import org.exist.util.ReadOnlyException;

public class FileLock {
    private static final Logger LOG = Logger.getLogger((Class)FileLock.class);
    private final long HEARTBEAT = 10100L;
    private static final byte[] MAGIC = new byte[]{101, 88, 105, 115, 116, 45, 100, 98};
    private BrokerPool pool;
    private File lockFile;
    private FileChannel channel = null;
    private final ByteBuffer buf = ByteBuffer.allocate(MAGIC.length + 8);
    private long lastHeartbeat = -1L;

    public FileLock(BrokerPool pool, String path) {
        this.pool = pool;
        this.lockFile = new File(path);
    }

    public FileLock(BrokerPool pool, File parent, String lockName) {
        this.pool = pool;
        this.lockFile = new File(parent, lockName);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean tryLock() throws ReadOnlyException {
        int attempt = 0;
        while (this.lockFile.exists()) {
            try {
                this.read();
            }
            catch (IOException e) {
                this.message("Failed to read lock file", null);
                e.printStackTrace();
            }
            if (!this.checkHeartbeat()) continue;
            if (++attempt == 1) {
                FileLock e = this;
                synchronized (e) {
                    try {
                        this.message("Waiting a short time for the lock to be released...", null);
                        this.wait(10200L);
                    }
                    catch (InterruptedException interruptedException) {
                        // empty catch block
                    }
                }
                try {
                    if (this.channel.isOpen()) {
                        this.channel.close();
                    }
                    this.channel = null;
                }
                catch (IOException e2) {}
                continue;
            }
            return false;
        }
        try {
            if (!this.lockFile.createNewFile()) {
                return false;
            }
        }
        catch (IOException e) {
            throw new ReadOnlyException(this.message("Could not create lock file", e));
        }
        try {
            this.save();
        }
        catch (IOException e) {
            throw new ReadOnlyException(this.message("Caught exception while trying to write lock file", e));
        }
        Properties params = new Properties();
        params.put(FileLock.class.getName(), this);
        this.pool.getScheduler().createPeriodicJob(10100L, new FileLockHeartBeat(this.lockFile.getName()), -1L, params);
        return true;
    }

    public void release() {
        try {
            if (this.channel.isOpen()) {
                this.channel.close();
            }
            this.channel = null;
        }
        catch (Exception e) {
            this.message("Failed to close lock file", e);
        }
        LOG.info((Object)("Deleting lock file: " + this.lockFile.getAbsolutePath()));
        this.lockFile.delete();
    }

    public Date getLastHeartbeat() {
        return new Date(this.lastHeartbeat);
    }

    public File getFile() {
        return this.lockFile;
    }

    private boolean checkHeartbeat() {
        long now = System.currentTimeMillis();
        if (this.lastHeartbeat < 0L || now - this.lastHeartbeat > 10100L) {
            this.message("Found a stale lockfile. Trying to remove it: ", null);
            this.release();
            return false;
        }
        return true;
    }

    private void open() throws IOException {
        RandomAccessFile raf = new RandomAccessFile(this.lockFile, "rw");
        this.channel = raf.getChannel();
    }

    protected void save() throws IOException {
        try {
            if (this.channel == null) {
                this.open();
            }
            long now = System.currentTimeMillis();
            this.buf.clear();
            this.buf.put(MAGIC);
            this.buf.putLong(now);
            this.buf.flip();
            this.channel.position(0L);
            this.channel.write(this.buf);
            this.channel.force(true);
            this.lastHeartbeat = now;
        }
        catch (NullPointerException npe) {
            if (this.pool.isShuttingDown()) {
                LOG.info((Object)"No need to save FileLock, database is shutting down");
            }
            throw npe;
        }
    }

    private void read() throws IOException {
        if (this.channel == null) {
            this.open();
        }
        this.channel.read(this.buf);
        this.buf.flip();
        if (this.buf.limit() < 16) {
            this.buf.clear();
            throw new IOException(this.message("Could not read file lock.", null));
        }
        byte[] magic = new byte[8];
        this.buf.get(magic);
        if (!Arrays.equals(magic, MAGIC)) {
            throw new IOException(this.message("Bad signature in lock file. It does not seem to be an eXist lock file", null));
        }
        this.lastHeartbeat = this.buf.getLong();
        this.buf.clear();
        DateFormat df = DateFormat.getDateInstance();
        this.message("File lock last access timestamp: " + df.format(this.getLastHeartbeat()), null);
    }

    protected String message(String message, Exception e) {
        StringBuffer str = new StringBuffer(message);
        str.append(' ').append(this.lockFile.getAbsolutePath());
        if (e != null) {
            str.append(": ").append(e.getMessage());
        }
        message = str.toString();
        if (LOG.isInfoEnabled()) {
            LOG.info((Object)message);
        }
        System.err.println(message);
        return message;
    }
}

